//
// Copyright (c) 2002
// Ronald Kevin Burton
//
// Z poniszym kodem nie jest zwizana adna gwarancja poprawnoci dziaania.
// Program zosta doczony do ksiki ".NET CLR. Ksiga eksperta" w celu
// ilustracji koncepcji i zasad przedstawionych w tej ksice. Program moe by 
// uywany na wasne ryzyko.
//
// Przyznaje si prawo do uycia lub kopiowania tego oprogramowania do dowolnego celu
// bez koniecznoci ponoszenia adnych opat pod warunkiem, e powysze uwagi zostan 
// zachowane we wszystkich kopiach. Przyznaje si take prawo do modyfikacji kodu
// i dystrybucji zmodyfikowanego kodu pod warunkiem zachowania powyszych uwag
// oraz doczenia informacji mwicej o modyfikacji kodu.
//
// 
using System;
using System.Diagnostics;

namespace LinkedList
{
	/// <summary>
	/// Podsumowanie dla LinkedObject.
	/// </summary>
	public class LinkedObject
	{
		private LinkedObject link;
		public LinkedObject()
		{
		}
		public LinkedObject(LinkedObject node)
		{
			link = node;
		}
		public LinkedObject Link
		{
			get
			{
				return link;
			}
			set
			{
				link = value;
			}
		}
	}
	public class FinalizedLinkedObject
	{
		private FinalizedLinkedObject link;
		public FinalizedLinkedObject()
		{
		}
		public FinalizedLinkedObject(FinalizedLinkedObject node)
		{
			link = node;
		}
		public FinalizedLinkedObject Link
		{
			get
			{
				return link;
			}
			set
			{
				link = value;
			}
		}
		~FinalizedLinkedObject()
		{
		}
	}
	// Kod wywodzi si z IDisposable, ci zapewnia implementacj
	// waciwych metod oraz umoliwia sprawdzenie, czy ten obiekt
	// obiekt obsuguje deterministyczny model finalizacji. Odbywa si
	// fto poprzez zapytanie obiektu o interfejs IDisposable.
	public class DisposeLinkedObject : IDisposable
	{
		// Stan tego obiektu powinien by umieszczony w buforze,
		// co daje pewno jednorazowej finalizacji obiektu. Naley upewni
		// si, czy ta zmienna zostaa poprawnie zainicjalizowana
		// we wszystkich konstruktorach tego obiektu lub czy zapewnia
		// domyln warto.
		private bool disposed = false;
		private DisposeLinkedObject link = null;
		// W tym kodzie zosta zaimplementowany tylko domylny konstruktor.
		// W przypadku prawdziwej aplikacji zasb kapsukowany przez ten
		// obiekt jest tutaj tworzony, istnieje inny konstruktor pozwalajcy
		// na waciwe utworzenie tego obiektu i/lub istnieje jawna metoda
		// open pozwalajca na utworzenie zasobu.
		public DisposeLinkedObject()
		{
		}
		public DisposeLinkedObject(DisposeLinkedObject node)
		{
			link = node;
		}
		public DisposeLinkedObject Link
		{
			get
			{
				return link;
			}
			set
			{
				link = value;
			}
		}
		// Bdzie to gwna metoda ujawniana przez obiekt w celu
		// deterministycznej finalizacji. Obiekt powinien by
		// otwierany poprzez dyrektyw new i konstruktora (lub przez 
		// jawne wywoanie open), a nastpnie jawnie zamknity
		// przez t metod.
		public void Close()
		{
			Dispose();
		}
		// Implementacja IDisposable. Istnieje tylko jedna metoda
		// implementacji w interfejsie IDisposable. Nie naley tworzy
		// metody wirtualnej, a klasa pochodna nie powinna mie
		// moliwoci zastpienia tej metody.
		public void Dispose()
		{
			// Usunicie zarzdzanych i niezarzdzanych zasobw.
			Dispose(true);
			// Usunicie tego obiektu z kolejki finalizacji, dziki
			// czemu kod finalizacji dla tego obiektu nie zostanie
			// wykonany po raz drugi.
			GC.SuppressFinalize(this);
		}
		// Funkcja Dispose(bool disposing) moe by uyta w dwch sytuacjach.
		// Jeli warto disposing jest prawd, to ta metoda zostaa wywoana
		// porednio lub bezporednio przez kod uytkownika. Usunicie
		// zarzdzanych i niezarzdzanych zasobw jest moliwe.
		// Jeli warto disposing jest faszem, to ta metoda zostaa wywoana
		// przez modu wykonawczy z wewntrz finalizera, przez co nie naley
		// odwoywa si do innych obiektw. Moliwe jest usunicie tylko
		// niezarzdzanych zasobw.
		protected virtual void Dispose(bool disposing)
		{
			// Sprawdzenie, czy wywoano ju Dispose.
			if(!this.disposed)
			{
				// Jeli warto disposing jest prawd, to usunicie wszystkich
				// zarzdzanych i niezarzdzanych zasobw.
				if(disposing)
				{
					// Usunicie zarzdzanych zasobw.
				}
				// Zwolnienie niezarzdzanych zasobw. Jeli warto disposing jest
				// faszem, to tylko ta cz kodu jest wykonywana.
				//
				// Uwaga: Ten kod nie zapewnia bezpieczestwa wtek.
				// Inny wtek moe rozpocz usuwanie obiektu po usuniciu
				// zarzdzanych zasobw, ale przed ustawieniem odpowiedniego
				// znacznika. Jeli zostanie uyty tu kod synchronizacji,
				// to nie mona pozwoli na blokowanie przez zbyt dugi czas 
				// ze wzgldu na moliwo wywoania tego kodu z poziomu
				// destruktora.
			}
			// Naley upewni si, czy zasoby nie zostan zwolnione
			// wicej ni jeden raz.
			disposed = true;         
		}

		// Ten destruktor zostanie uruchomiony tylko jeli metoda Dispose 
		// nie zostanie wywoana, gdy w przeciwnym wypadku nastpuje uycie
		// GC.SupressFinalize(), ktry sprawdza fakt wywoania destruktora
		// dla tego obiektu. Daje to szans finalizacji klasy bazowej.
		// Nie naley uywa destruktorw dla typw wywodzcych si z tej klasy.
		~DisposeLinkedObject()      
		{
			// Nie naley tworzy tu ponownie kodu czyszczcego Dispose.
			// Wywoanie Dispose(false) jest optymalne w zakresie czytelnoci
			// i atwoci obsugi.
			// Wywoanie Dispose(false) powoduje usunicie tylko niezarzdzanych
			// zasobw.
			Dispose(false);
		}

	}
}
